<?php
/**
 * License limited to a single site, for use on another site please purchase a license for this module.
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/osl-3.0.php
 *
 * @author    Dingedi.com
 * @copyright Copyright 2025 © Dingedi All right reserved
 * @license   http://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 *
 * @category  Dingedi PrestaShop Modules
 */
if (!defined('_PS_VERSION_')) {
    exit;
}

class DgMailsFinder
{
    /**
     * @var mixed[]
     */
    private $langFrom;

    /**
     * @throws Exception
     */
    public function __construct(array $langFrom)
    {
        $this->langFrom = $langFrom;
    }

    /**
     * @param string $path
     * @param bool $searchModule
     * @return mixed[]
     */
    public function find($path, $searchModule = false)
    {
        if ($searchModule) {
            return $this->findMailsModulesFiles($path);
        } else {
            return $this->findMailsFiles($path);
        }
    }

    /**
     * @param string $path
     * @return mixed[]
     */
    private function findMailsFiles($path)
    {
        $path = (string) $path;
        $path = rtrim($path, '/') . '/';

        return $this->findFiles($path . $this->langFrom['iso_code']);
    }

    /**
     * @param string $path
     * @return mixed[]
     */
    private function findMailsModulesFiles($path)
    {
        $path = (string) $path;
        $mails = [];
        foreach (Module::getModulesInstalled() as $module) {
            if (file_exists($path . $module['name'] . '/mails')) {
                $mails[] = [
                    'name' => $module['name'],
                    'mails' => $this->findMailsFiles($path . $module['name'] . '/mails/'),
                ];
            }
        }

        return array_values($mails);
    }

    /**
     * @param string $path
     * @return mixed[]
     */
    private function findFiles($path)
    {
        $path = (string) $path;
        if (!file_exists($path)) {
            return [];
        }

        $mails = [];
        $recursiveDirectoryIterator = new RecursiveDirectoryIterator($path);
        $extensions = ['html', 'txt'];

        /** @var SplFileInfo $file */
        foreach (new RecursiveIteratorIterator($recursiveDirectoryIterator) as $file) {
            if (in_array($file->getExtension(), $extensions)) {
                $name = basename($file->getFilename(), '.' . $file->getExtension());

                $mail = [
                    'path' => $file->getPath() . '/' . $file->getFilename(),
                    'filename' => $file->getFilename(),
                    'extension' => $file->getExtension(),
                    'translations' => [],
                ];

                foreach (Dingedi\PsTools\DgShopInfos::getLanguages() as $lang) {
                    if ((int) $lang['id_lang'] === (int) $this->langFrom['id_lang']) {
                        continue;
                    }

                    $source = Tools::file_get_contents($mail['path']);
                    $langFile = Tools::file_get_contents(dirname(dirname($mail['path'])) . '/' . $lang['iso_code'] . '/' . $mail['filename']);

                    if ($source && $langFile === false) {
                        $translated = false;
                    } elseif ($source && $langFile) {
                        $translated = $source !== $langFile;
                    } else {
                        continue;
                    }

                    $mail['translations'][] = [
                        'id_lang' => (int) $lang['id_lang'],
                        'name' => $lang['name'],
                        'locale' => Tools::strtoupper($lang['iso_code']),
                        'translated' => $translated,
                    ];
                }

                $search = array_search($name, array_column($mails, 'name'));

                if ($search === false) {
                    $mails[] = [
                        'name' => $name,
                        'mails' => [$mail],
                    ];
                } else {
                    $mails[$search]['mails'][] = $mail;
                }
            }
        }

        usort($mails, function ($a, $b) {
            $battleShipcompare = function ($left, $right) {
                if ($left === $right) {
                    return 0;
                }
                return $left < $right ? -1 : 1;
            };
            return $battleShipcompare($a['name'], $b['name']);
        });

        return array_values($mails);
    }
}
